package gov.va.med.mhv.vitals.web.controller;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.vitals.dto.BloodPressureReadingDTO;
import gov.va.med.mhv.vitals.enums.PeriodEnumeration;
import gov.va.med.mhv.vitals.util.CommonUtility;
import gov.va.med.mhv.vitals.web.util.WebUtility;

import java.awt.Color;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.chart.plot.Marker;
import org.jfree.chart.plot.ValueMarker;
import org.jfree.chart.plot.XYPlot;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.jfree.ui.RectangleAnchor;
import org.jfree.ui.TextAnchor;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.primefaces.model.StreamedContent;
import org.primefaces.model.chart.AxisType;
import org.primefaces.model.chart.DateAxis;
import org.primefaces.model.chart.LineChartModel;
import org.primefaces.model.chart.LineChartSeries;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

@ManagedBean
@Component
@Scope("session")
public class BloodPressureController extends AbstractController {

	private static final long serialVersionUID = 3674775257581021572L;
	private static Logger log = LogManager.getLogger(BloodPressureController.class);
	private List<BloodPressureReadingDTO> bpreadings = new ArrayList<BloodPressureReadingDTO>();
	private BloodPressureReadingDTO selectedBPReading = new BloodPressureReadingDTO();
	private BloodPressureReadingDTO newBPReading = new BloodPressureReadingDTO();
	private static final String  TITLE ="Blood Pressure";
	private static final String  XAXIS ="Date";
	private static final String  YAXIS ="mmHg";
	private static final String  SERIES1="Systolic";
	private static final String  SERIES2="Diastolic";
	
	
	public void init(ComponentSystemEvent event) {
		findUser();
		DataTable bpTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("bloodPressureForm:bpressureList");
		userprofileId = getUserProfileId();
		
		if(!FacesContext.getCurrentInstance().isPostback()){
			resetMessages();
			if (userprofileId != null) {
				bpreadings = findBPReadings(userprofileId);
			}
			setRowsPerPage(10);
			setPeriod(PeriodEnumeration.ONE_MONTH.getDescription());
		}
		else{
			if(sortColumn != null && sortBy != null){
				bpTable.setValueExpression("sortBy", sortColumn);
				bpTable.setSortOrder(sortBy);
			}
		}
		getJchart();
	}
	

	public void onSort(SortEvent event){
		sortColumn=event.getSortColumn().getValueExpression("sortBy");
		sortBy=event.isAscending()?"ascending":"descending";
	}

	private List<BloodPressureReadingDTO> findBPReadings(Long userProfileId) {
		List<BloodPressureReadingDTO> dtoList = null;
		try {
			
			dtoList = this.vitalSignsService.getBloodPressureReadingsForUser(userProfileId);

		} catch (Exception e) {
			log.error("Error in Find BP readings:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR,ERR_PRCS_RQST,ERR_PRCS_RQST));
		}
		
		return dtoList;
	}

	public String showDetail(BloodPressureReadingDTO bloodPressureReadingDTO) {
		resetMessages();
		selectedBPReading = bloodPressureReadingDTO;
		setHourMinute(selectedBPReading);
		return "bloodPressure";
	}
	
	public String showDahBoardDetail() {
		resetMessages();
		findUser();
		String bpid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("bpid");
		if (bpid != null) {
			Long id = Long.valueOf(bpid);
			Long userProfileId = getUserProfileId();
			selectedBPReading=getBPReadingById(userProfileId, id);
		}	
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().
		           handleNavigation(context, null, "/views/bloodpressure/bloodPressure.xhtml");
		return null;
	}

	
	private BloodPressureReadingDTO getBPReadingById(Long userProfileId, Long id) {
		try {
			
			return this.vitalSignsService.getBloodPressureReadingById(userProfileId, id);
			
		} catch (Exception e) {
			log.error("Error in Find BP reading By id:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR,ERR_PRCS_RQST,ERR_PRCS_RQST));
		}
		
		return null;
	}


	public String editDisplay(BloodPressureReadingDTO bloodPressureReadingDTO) {
		resetMessages();
		selectedBPReading = bloodPressureReadingDTO;
		setHourMinute(selectedBPReading);
		return "editBPDisplay";
	}
	
	public String resetEditDisplay(){
		return "editBPDisplay";
	}

	public String deleteDisplay(BloodPressureReadingDTO bloodPressureReadingDTO) {
		resetMessages();
		selectedBPReading = bloodPressureReadingDTO;
		setHourMinute(selectedBPReading);
		setDeleteOrigin("tableView");
		return "deleteBPDisplay";
	}
	
	public String deleteRecordDisplay() {
		resetMessages();
		setDeleteOrigin(null);
		return "deleteBPDisplay";
	}

	
	private void setHourMinute(BloodPressureReadingDTO  selectedBPReading){
		String dateTime =WebUtility.dateToString(selectedBPReading.getReading(),HOUR_MINUTE);
		if(!dateTime.equals(DEFAULT_DATETIME)){
			selectedBPReading.setHour(WebUtility.dateToString(selectedBPReading.getReading(), HOURS));
			selectedBPReading.setMinute(WebUtility.dateToString(selectedBPReading.getReading(), MINUTES));
		}

	}

	public String addDisplay() {
		resetMessages();
		newBPReading=new BloodPressureReadingDTO();
		newBPReading.setReading(new Date());
		return "addBPDisplay";
	}
	
	
	public String dashboardAddDisplay() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			bpreadings = findBPReadings(userprofileId);
		}
		newBPReading=new BloodPressureReadingDTO();
		newBPReading.setReading(new Date());
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().
		           handleNavigation(context, null, "/views/bloodpressure/addBPDisplay.xhtml");
		return null;
	}
	
	public String dashboardViewMore() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			bpreadings = findBPReadings(userprofileId);
		}
		return "/views/bloodpressure/bloodPressureList";
	}


	public String delete() {
		String bpid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("bpid");
		try{
			
			this.vitalSignsService.deleteBloodPressureReading(getUserProfileId(), Long.valueOf(bpid));
			bpreadings = findBPReadings(userprofileId);
			deleteMessage = true;
			return "bloodPressureList";
			
		} catch(Exception e){
			log.error("Failed to Delete BP record",e);
			
			FacesContext.getCurrentInstance().addMessage(null, 
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}
		return null;
	}

	public String save() {
		String outcome=null;
		try{
			Long bpId =saveBPReading();
			if(bpId != null && bpId > 0) {
				outcome="bloodPressureList";
				saveMessage = true;
				bpreadings = findBPReadings(userprofileId);
			}
		} catch( MHVException e){
			log.error("Failed to Save BP record",e);
			
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public Long saveBPReading() throws MHVException {
		resetMessages();
		String bpid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("bpid");
		Long id=null;
		BloodPressureReadingDTO bpreading=null;
		
		if (bpid != null) {
			id = Long.valueOf(bpid);
			bpreading=getSelectedBPReading();
		} else {
			bpreading=getNewBPReading();
		}
		
		bpreading = prepareBPReading(id,bpreading);
		Long bpId = this.vitalSignsService.saveBloodPressureReading(bpreading);
		bpreading.setBloodPressureId(bpId);
		
		return bpId;
	}

	private BloodPressureReadingDTO prepareBPReading(Long id,BloodPressureReadingDTO bpreading) {
		Date dateTime=null;
		if(bpreading.getReading() != null ){
			if(bpreading.getHour() == null  && bpreading.getMinute() == null){
			   dateTime = WebUtility.getDaTeTime(bpreading.getReading(), DEFAULT_HOUR, DEFAULT_MINUTE);
			}else{
			   dateTime = WebUtility.getDaTeTime(bpreading.getReading(), bpreading.getHour(), bpreading.getMinute());	
			}
		}
		bpreading.setReading(dateTime);
		bpreading.setUserprofileId(getUserProfileId());
		return bpreading;
	}

	public String saveAndAdd() {
		String outcome=null;
		try{
			Long bpId=saveBPReading();
			if(bpId != null && bpId > 0){
				outcome="addBPDisplay";
				saveAndAddMessage = true;
				bpreadings = findBPReadings(userprofileId);
				newBPReading=new BloodPressureReadingDTO();
				newBPReading.setReading(new Date());
			}

		}catch( MHVException e){
			log.error("Failed to Save BP record",e);
			
			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}
	
	public String updateLineModel(){
        return "bloodPressureGraph";
	}
	
	
	public LineChartModel   getLineModel(){
		LineChartModel dateModel = new LineChartModel();
	    LineChartSeries series1 = new LineChartSeries();
	    LineChartSeries series2 = new LineChartSeries();
	    Set<String> uniqueYears = new HashSet<String>();
	    String period = getPeriod();
	    Date startDate=null;
	    List<BloodPressureReadingDTO> graphList;
	    if(!period.equals(PeriodEnumeration.ALL_DATA.getDescription())){
	    	startDate =getStartGraphDate(period);
	    }
	    if(startDate != null){
	       graphList = getGraphlist(bpreadings,startDate);
	    }else{
	       graphList = bpreadings;
	    }
	    for( BloodPressureReadingDTO bpreading : graphList){
	    	series1.set(CommonUtility.dateToString(bpreading.getReading(),YYYY_MM_DD_HHMMSS_FORMAT), bpreading.getDiastolic());
	    	uniqueYears.add(CommonUtility.dateToString(bpreading.getReading(),YYYY_MM_DD_HHMMSS_FORMAT).substring(0,4));
	    }
	    for( BloodPressureReadingDTO bpreading : graphList){
	    	series2.set(CommonUtility.dateToString(bpreading.getReading(),YYYY_MM_DD_HHMMSS_FORMAT), bpreading.getSystolic());
	    }
	    dateModel.addSeries(series1);
	    dateModel.addSeries(series2);
		dateModel.getAxis(AxisType.Y).setLabel("BPM");
		DateAxis axis = new DateAxis("DATE");
		axis.setTickAngle(-90);
		setTickValues(axis,period,uniqueYears);
		dateModel.getAxes().put(AxisType.X, axis);  
		dateModel.setExtender("chartExtender");
        return dateModel;
	}
	
	
	public StreamedContent getJchart(){
		StreamedContent content=null;
		try{
			content =createChart(TITLE,XAXIS,YAXIS);
		}catch(Exception e){
			log.error("Failed to create Chart",e);
		}
		return content;
	}
	
	
	protected void setValueRanges(XYPlot plot) {
		Marker diastHigh= new ValueMarker(80);
		Marker diastHyp= new ValueMarker(90);
		Marker systHigh= new ValueMarker(120);
		diastHigh.setPaint(Color.black);
		diastHyp.setPaint(Color.black);
		systHigh.setPaint(Color.black);
		diastHigh.setLabel("Diast High");
		diastHyp.setLabel("Diast Hyp");
		systHigh.setLabel("Syst High");
		diastHigh.setLabelAnchor(RectangleAnchor.BOTTOM_RIGHT);
		diastHyp.setLabelAnchor(RectangleAnchor.BOTTOM_RIGHT);
		systHigh.setLabelAnchor(RectangleAnchor.BOTTOM_RIGHT);
		diastHigh.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
		diastHyp.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
		systHigh.setLabelTextAnchor(TextAnchor.TOP_RIGHT);
		plot.addRangeMarker(diastHigh);
		plot.addRangeMarker(diastHyp);
		plot.addRangeMarker(systHigh);
	}


	
	protected XYDataset createDataset(Set<String> uniqueYears) {
		 TimeSeriesCollection dataset = new TimeSeriesCollection();
		 TimeSeries series1 = new TimeSeries(SERIES1);
		 TimeSeries series2 = new TimeSeries(SERIES2);
		 String period = getPeriod();
		 Date startDate=null;
		 List<BloodPressureReadingDTO> graphList;
		 if(!period.equals(PeriodEnumeration.ALL_DATA.getDescription())){
		     startDate =getStartGraphDate(period);
		 }
		 if(startDate != null){
		     graphList = getGraphlist(bpreadings,startDate);
		 }else{
		     graphList = bpreadings;
		 }
	     for( BloodPressureReadingDTO bpreading : graphList){
	    	 
	    	 series2.addOrUpdate(new Day(bpreading.getReading()),bpreading.getDiastolic());
	    	 series1.addOrUpdate(new Day(bpreading.getReading()),bpreading.getSystolic());
		     uniqueYears.add(CommonUtility.dateToString(bpreading.getReading(),YYYY_MM_DD_HHMMSS_FORMAT).substring(0,4));
		     
	     }
		 dataset.addSeries(series1);
		 dataset.addSeries(series2);
		 return dataset;
	}

	


	private List<BloodPressureReadingDTO> getGraphlist(List<BloodPressureReadingDTO> bpeadings, Date startDate) {
		List<BloodPressureReadingDTO> graphList= new ArrayList<BloodPressureReadingDTO>();
		Integer startdateValue = Integer.valueOf(CommonUtility.dateToString(startDate, YYYYMMDD_FORMAT));
		for(BloodPressureReadingDTO bpreading : bpeadings){
			Integer readingValue =Integer.valueOf(CommonUtility.dateToString(bpreading.getReading(), YYYYMMDD_FORMAT));
			if(readingValue >= startdateValue){
				graphList.add(bpreading);
			}
		}
		return graphList;
	}
	
	public String printerFriendlySummary(){
		resetMessages();
		return "printBPSummary";
	}

	public List<BloodPressureReadingDTO> getBpreadings() {
		return bpreadings;
	}

	public void setBpreadings(List<BloodPressureReadingDTO> bpreadings) {
		this.bpreadings = bpreadings;
	}

	public BloodPressureReadingDTO getSelectedBPReading() {
		return selectedBPReading;
	}

	public void setSelectedBPReading(BloodPressureReadingDTO selectedBPReading) {
		this.selectedBPReading = selectedBPReading;
	}

	public BloodPressureReadingDTO getNewBPReading() {
		return newBPReading;
	}

	public void setNewBPReading(BloodPressureReadingDTO newBPReading) {
		this.newBPReading = newBPReading;
	}

}
